; ===========================================================================
; ---------------------------------------------------------------------------
; GSM 6.10 Z80 driver - MarkeyJester
; ---------------------------------------------------------------------------
; This experiment was only able to output a sample rate of 100Hz.  While it
; works 100% to the original specifications, it unfortunately is not practical
;
; ideally 13.195kHz (25% clock rate of YM) would be needed, about just over
; 135 Z80 CPU T-state cycles (my hardware produced 13.156kHz before the
; capacitors lag behind for the filter effect).
; ---------------------------------------------------------------------------

;	Please see GSM 6.10 pdf file for specifications.
;
;	The bit format is extracted/converted out mostly through swap
;	tables, with exception to those that were the same or slower than
;	shift/rotate instructions.
;
;	Many range conditions were performed by adding the negative constant
;	rather than subtracting as positive, as there is no sub for hl, only
;	sbc which is 4 cycles slower than add.
;
;	Multiplication is performed by an unrolled series of carry controlled
;	arithmetic (the same kind more modern CPU's do internally).  With
;	optimisation assumptions (for example, no point shifting the current
;	result up at the beginning when it's 0, and we haven't found a carry
;	yet).
;
;	The multiplication subroutine is "mult_r" and performs the same rounding
;	and right shift GSM requires.  It's called via rst.
;
;	Samples are flushed out of the same buffer GSM is decoded into as
;	flushing out is faster than decoding, so no double buffer is required.
;
;	The YM2612 Timer B has been used (it's slower than A), to ensure consistent
;	playback, however, the multiplications are too slow, so it ends up taking
;	too long to hear anything meaningful, but it does work and is consistent.
;	Again... called via rst.

; ===========================================================================
; ---------------------------------------------------------------------------
; Start of Z80 ROM/RAM
; ---------------------------------------------------------------------------

Start:
		di						;  4	; disable interrupts
		ld	sp,02000h				; 10	; setup stack address
		jp	Init					; 10	; jump into loop start

; ===========================================================================
; ---------------------------------------------------------------------------
; YM2612 flush
; ---------------------------------------------------------------------------

		align	8

TimerGSM =	$E1	; $FF = fastest/shortest | $00 = slowest/longest

Flush:
		exx						;  4	; swap to flush registers
		ex	af,af					;  4	; store a

	; --- Rate finder ---
	; Every time "Flush" is called, it must wait for Timer A
	; before flushing a sample, if Timer A is already overflown
	; then the Z80 was late to flushing, meaning the timer is
	; too fast (too short), and must be made longer.  The ideal
	; timer is one where the Z80 always gets here before it
	; finishes.  This 1 in "a" will not be cleared if the Z80
	; was late.
	; -------------------

		ld	a,002h					;  7	; setup rate finder flag
	.Wait:	ld	c,(hl)					;  7	; load flags
		zand	c					;  4	; clear rate finder flag (read above)
		sra	c					;  8	; push timer B overflow flag into the carry
		sra	c					;  8	; ''
		jr	nc,.Wait				;  7|12	; if timer isn't finished, loop
		zor	a					;  4	; check rate finder flag
	.Halt:	jr	nz,.Halt				;  7|12	; if Z80 was late, it'll get stuck here...

		ld	(hl),02Ah				; 10	; set YM2612 to DAC port
		inc	l					;  4	; advance to data port
		ld	a,(de)					;  7	; load sample
		ld	(hl),a					;  7	; save sample
		dec	l					;  4	; mvoe back to address port
		inc	e					;  4	; advance to next sample

		ld	(hl),027h				;  7	; set YM2612 to timer flags port
		inc	l					;  4	; advance to data port
		ex	af,af					;  4	; restore a
		ld	(hl),00101010b				; 10	; enable timer B and trigger overflow bit
		dec	l					;  4	; move back to address port
		exx						;  4	; restore GSM registers
		ret						; 10	; done

; ===========================================================================
; ---------------------------------------------------------------------------
; LAR reflection, store, and end check
; ---------------------------------------------------------------------------
; The reflection of LAR for 4 different frames, are summed differently, however
; the reflection itself, the storage of that reflection, and the loop checks
; are identical, these are stored in a call to here to save space
; ---------------------------------------------------------------------------

		align	8

ReflEndLAR:
ReflectRET:	jp	ReflectLAR				; 10	; convert to rp
		ex	de,hl					;  4	; store rp in de
		ld	h,rrp>>8				;  7	; load rrp table
		ld	l,a					;  4	; set i index
		ld	(hl),e					;  7	; save rp to table
		inc	l					;  4	; ''
		ld	(hl),d					;  7	; ''

		zadd	2					;  7	; increase i x 2
		cp	(rrp&0FFh)+8*2				;  7	; have 1 to 8 been done?
		ret						; 10	; return to reflection frame routine

; ===========================================================================
; ---------------------------------------------------------------------------
; mult_r (GSM multiplication with rounding)
; ---------------------------------------------------------------------------
; This GSM function performs a multiplication of two WORD values, to create
; a LONGWORD result, however, the WORD values are 15 bit only, so the LONGWORD
; format ends up being (binary): XQQQQQQQ QQQQQQQQ QFFFFFFF FFFFFFFF
;
; X = unused
; Q = quotient
; F = fraction
;
; The quotient is rounded up/down based on the fraction, and only the quotient
; is returned QQQQQQQQ QQQQQQQQ as a WORD result:
;
; WORD Out = (((LONG) A * B) + 0x4000) >> 0x0F;
; ---------------------------------------------------------------------------
multst	macro	lab1, lab2
	\lab1:
		ex	de,hl					;  4	; swap for hl powers
		ld	h,b					;  4	; add multiplier to lower word result
		ld	l,c					;  4	; always same as a move since result starts as 0
		jp	\lab2					; 10	; jump to actual multiply chain
	endm
; ---------------------------------------------------------------------------
mult	macro	lab2, lab3
	\lab2:
		zadd	hl,hl					; 11	; get next bit (put MSB on carry)
		rl	e					;  8	; shift multiplicand up (and shift upper word in)
		rl	d					;  8	; shfit multiplicand up (and shift upper word in)
		jr	nc,\lab3				;  7|12	; if no carry, branch
		zadd	hl,bc					; 11	; add multiplier to lower word result
		jr	nc,\lab3				;  7|12	; if not overflown, branch
		inc	de					;  6	; carry the 1 to the upper word result
	endm
; ---------------------------------------------------------------------------

		align	8

	; --- signed multiplication ---

mult_r:
		; hl = multiplicand x bc = multiplier

		xor	a					;  4	; clear negate flags
		bit	7,h					;  8	; is the multiplicand negative?
		jr	z,.noneghl				;  7|12	; if not, skip
		ex	de,hl					;  4	; negate hl
		ld	h,a					;  4	; clear ready for subtractive negation
		ld	l,a					;  4	; ''
		sbc	hl,de					; 16	; '' (carry already clear by xor a)
		ccf						;  4	; '' (convert carry back to clear the sbc will always set in this circumstance)
		inc	a					;  4	; mark hl as negated

	.noneghl:
		bit	7,b					;  8	; is the multiplier negative?
		jr	z,.nonegbc				;  7|12	; if not, skip
		ex	de,hl					;  4	; store hl
		ld	hl,00000h				; 10	; clear ready for subtractive negation
		sbc	hl,bc					; 16	; '' (carry already clear)
		ld	b,h					;  4	; copy back to bc
		ld	c,l					;  4	; ''
		ex	de,hl					;  4	; restore hl
		inc	a					;  4	; mark bc as negated

	.nonegbc:

	; --- unsigned multiplication ---

u_mult_r:
		ld	de,00000h				; 10	; reset result
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mc0					;  7|12	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mc1					;  7|12	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mc2					;  7|12	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mc3					;  7|12	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mc4					;  7|12	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mc5					;  7|12	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mc6					; 10	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mc7					; 10	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mc8					; 10	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mc9					; 10	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mcA					; 10	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mcB					; 10	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mcC					; 10	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mcD					; 10	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jr	c,.mcE					; 10	; if bit is found, branch into calculation
		zadd	hl,hl					; 11	; shift multiplicand up
		jp	nc,.mcdn				; 10	; if bit is found, branch into calculation
		ld	d,b					;  4	; add multiplier to lower word result
		ld	e,c					;  4	; always same as a move since result starts as 0
		jp	.mcdn					; 10	; done...

		multst	.mc0, .mcn1
		multst	.mc1, .mcn2
		multst	.mc2, .mcn3
		multst	.mc3, .mcn4
		multst	.mc4, .mcn5
		multst	.mc5, .mcn6
		multst	.mc6, .mcn7
		multst	.mc7, .mcn8
		multst	.mc8, .mcn9
		multst	.mc9, .mcnA
		multst	.mcA, .mcnB
		multst	.mcB, .mcnC
		multst	.mcC, .mcnD
		multst	.mcD, .mcnE
		multst	.mcE, .mcnF

		mult	.mcn1, .mcn2
		mult	.mcn2, .mcn3
		mult	.mcn3, .mcn4
		mult	.mcn4, .mcn5
		mult	.mcn5, .mcn6
		mult	.mcn6, .mcn7
		mult	.mcn7, .mcn8
		mult	.mcn8, .mcn9
		mult	.mcn9, .mcnA
		mult	.mcnA, .mcnB
		mult	.mcnB, .mcnC
		mult	.mcnC, .mcnD
		mult	.mcnD, .mcnE
		mult	.mcnE, .mcnF
		mult	.mcnF, .mcndn

	.mcndn:
		ex	de,hl					;  4	; swap hl back

	.mcdn:

	;	hl = upper word
	;	de = lower word

	; --- GSM specific >> F ---

		rrca						;  4	; check if hl or bc independently were negated
		jr	nc,.NoNeg				;  7|12	; if not, skip over negation
		cp	a					;  4	; clear carry
		push	hl					; 11	; store positive upper word result
		ld	hl,00000h				; 10	; get negative lower word result
		sbc	hl,de					; 15	; ''
		ld	a,h					;  4	; store upper byte of lower word (needed for rounding later)
		pop	de					; 10	; restore upper word result
		ld	hl,00000h				; 10	; get negative upper word result (with carry from lower word negation)
		sbc	hl,de					; 15	; ''
		ld	d,a					;  4	; restore upper byte of lower word (for rounding below)

	.NoNeg:
		rl	d					;  8	; push MSB of lower word into carry
		jp	m,.Round				; 10	; if the quotient needs rounding up, branch
		adc	hl,hl					; 11	; this is the >> F, done by basically doing a << 1 (with the carry of the lower word d)
		cp	a					;  4	; clear carry
		ret						; 10	; done

	.Round:
		adc	hl,hl					; 11	; this is the >> F, done by basically doing a << 1 (with the carry of the lower word d)
		inc	hl					;  6	; round up...
		cp	a					;  4	; clear carry
		ret						; 10	; done

; ===========================================================================
; ---------------------------------------------------------------------------
; Init/setup
; ---------------------------------------------------------------------------

Init:
		im	1					;  8	; set interrupt mode

		ld	hl,04000h				; 10	; load YM2612 address port

		ld	(hl),02Bh				; 10	; set address to DAC switch
		inc	l					;  4	; advance to data port
		znop			; might not need this	;  4	; delay
		ld	(hl),080h				; 10	; enable DAC
		dec	l					;  4	; move back to address port
		znop			; might not need this	;  4	; delay

		ld	(hl),026h				; 10	; set address to timer A lower
		inc	l					;  4	; advance to data port
		znop			; might not need this	;  4	; delay
		ld	(hl),TimerGSM				; 10	; write tiemr A lower
		dec	l					;  4	; move back to address port
		znop			; might not need this	;  4	; delay

		ld	(hl),027h				; 10	; set address to timer port
		inc	l					;  4	; advance to data port
		znop			; might not need this	;  4	; delay
		ld	(hl),00101010b				; 10	; enable timer A and trigger overflow bit
		dec	l					;  4	; move back to address port

		exx						;  4	; store flush registers

; ---------------------------------------------------------------------------
; GSM data request play loop
; ---------------------------------------------------------------------------

WaitData:
		ld	hl,06001h				; 10	; load bank switch port
		db	74h|((GSM_Sample>>0Fh)&1)		;  7	; 0000 0000 1  32KB -  64KB (  8000 -   10000)
		db	74h|((GSM_Sample>>10h)&1)		;  7	; 0000 0001 0  64KB - 128KB ( 10000 -   20000)
		db	74h|((GSM_Sample>>11h)&1)		;  7	; 0000 0010 0 128KB - 256KB ( 20000 -   40000)
		db	74h|((GSM_Sample>>12h)&1)		;  7	; 0000 0100 0 256KB - 512KB ( 40000 -   80000)
		db	74h|((GSM_Sample>>13h)&1)		;  7	; 0000 1000 0 512KB -   1MB ( 80000 -  100000)
		db	74h|((GSM_Sample>>14h)&1)		;  7	; 0001 0000 0   1MB -   2MB (100000 -  200000)
		db	74h|((GSM_Sample>>15h)&1)		;  7	; 0010 0000 0   2MB -   4MB (200000 -  400000)
		db	74h|((GSM_Sample>>16h)&1)		;  7	; 0100 0000 0   4MB -   8MB (400000 -  800000)
		db	74h|((GSM_Sample>>17h)&1)		;  7	; 1000 0000 0   8MB -  16MB (800000 - 1000000)

		ld	hl,08000h				; 10	; window where GSM data is

DecodeBlock:
		ld	a,(hl)					;  7	; load start of next block
		zor	a					;  4	; check if it's negative
		jp	p,WaitData				; 10	; if not, wait (GSM blocks start with D#)

		exx						;  4	; get flush registers
		ld	de,Output				; 10	; reset buffer
		exx						;  4	; get GSM registers

; ---------------------------------------------------------------------------
; Loading and decoding of the quantised Log.-Area Ratio coefficients (LARc)
; LAR''(i) = ( LARc(i) - B(i) )/ A(i)
; --- expected registers ----------------------------------------------------
; hl = Start of GSM block:
;
;	DDDD1111
;	11222222
;	33333444
;	44555566
;	66777888
; ---------------------------------------------------------------------------

	; 48 cycles

SaveLARppLast	macro	Address
		ld	c,a					;  4	; load high LAR''(1)
		ld	a,(bc)					;  7	; ''
		ld	(\Address+0),a				; 13	; save to LARpp(1)
		inc	b					;  4	; load low LAR''(1)
		ld	a,(bc)					;  7	; ''
		ld	(\Address+1),a				; 13	; save to LARpp(1)
		endm

	; 52 cycles

SaveLARpp	macro	Address
		SaveLARppLast	\Address
		dec	b					;  4	; go back for LAR''(2)
		endm

	;  645 cycles

LoadLARc	macro	Address
		ld	e,(hl)					;  7	; load DDDD1111
		inc	hl					;  6	; next byte
		ld	a,(de)					;  7	; load 00111100
		ld	d,ASR6M00000011>>8			;  7	; next table
		ld	e,(hl)					;  4	; load 11222222
		inc	hl					;  6	; next byte
		ex	de,hl					;  4	; swap for hl powers
		zadd	(hl)					;  7	; make 00111111
		SaveLARpp	\Address+0			; 52	; save to LARpp table
		ld	a,l					;  4	; load 00222222
		zand	00111111b				;  7	; ''
		SaveLARpp	\Address+2			; 52	; save to LARpp table
		ex	de,hl					;  4	; get back GSM data
		ld	d,ASR3M00011111>>8			;  7	; next table
		ld	e,(hl)					;  7	; load 33333444
		inc	hl					;  6	; next byte
		ld	a,(de)					;  7	; load 00033333
		inc	d					;  4	; next table
	rst	Flush
		SaveLARpp	\Address+4			; 52	; save to LARpp table
		ld	a,(de)					;  7	; load 00044400
		ld	e,(hl)					;  7	; load 44555566
		inc	hl					;  6	; next byte
		ld	d,ASR6M00000011>>8			;  7	; '' (load table for it)
		ex	de,hl					;  4	; swap for hl powers
		zadd	(hl)					;  7	; make 00044444
		zadd	LARpp4&0FFh				;  7	; ASR6M00000011 has no add parameter
		SaveLARpp	\Address+6			; 52	; save to LARpp table
		ld	h,ASR2M00001111>>8			;  7	; go back to recent table
		ld	a,(hl)					;  4	; load 00005555
		SaveLARpp	\Address+8			; 52	; save to LARpp table
		ex	de,hl					;  4	; get back GSM data
		inc	d					;  4	; next table
		ld	a,(de)					;  7	; load 00006600
		ld	e,(hl)					;  7	; load 66777888
		ld	d,ASR6M00000011>>8			;  7	; '' (load table for it)
		ex	de,hl					;  4	; swap for hl powers
		zadd	(hl)					;  7	; make 00006666
		zadd	LARpp6&0FFh				;  7	; ASR6M00000011 has no add parameter
		SaveLARpp	\Address+10			; 52	; save to LARpp table
	rst	Flush
		inc	h					;  4	; next table
		ld	a,(hl)					;  4	; load 00000777
		zadd	LARpp7&0FFh				;  7	; no add parameter
		SaveLARpp	\Address+12			; 52	; save to LARpp table
		ld	a,l					;  4	; load 66777888
		zand	00000111b				;  7	; make 00000888
		zadd	LARpp8&0FFh				;  7	; no add parameter (since no table)
		SaveLARppLast	\Address+14			; 48	; save to LARpp table
		ex	de,hl					;  4	; get back GSM data
		inc	hl					;  6	; advance to next byte
		endm

; ---------------------------------------------------------------------------
; routine itself
; ---------------------------------------------------------------------------

	rst	Flush
		ld	b,LARpp1n2>>8				;  7	; setup LARc to LARpp swap tables
		ld	d,LARcTables>>8				;  7	; load LARc arithmetic swap table list
		ld	a,(LARcur)				; 13	; load which table is current
		cp	LARppj&0FFh				;  7	; is the current table in LARppj?
		jp	nz,.ppjm				; 10	; if not, current table is in LARppjm

	.ppj:	LoadLARc  LARppj				;642	; load LARc, convert to LAR', and save to LAR'' table
		jp	.DoneLAR				; 10	; finish

	.ppjm:	LoadLARc  LARppjm				;642	; load LARc, convert to LAR', and save to LAR'' table

	.DoneLAR:

; ---------------------------------------------------------------------------
; frames 1 to 4 (i.e. 3, 2, 1, 0)
; ---------------------------------------------------------------------------

		ld	a,Output&0FFh				;  7	; reset output buffer address
		ld	(Buffer+1),a				; 13	; ''

		ld	a,3					;  7	; reset frame counter
		ld	(frame+1),a				; 13	; ''

		ld	ix,dfTable				; 14	; reset df table underflow access (-160 to 0)
		ld	iy,dpTable				; 14	; reset df table underflow access (0 to 160)

	rst	Flush

FrameLoop:

	; ---------------------------------------------------------------------------
	; Unpacking Nc, bc, Mc, xmaxc
	; ---------------------------------------------------------------------------
	; LTP lag (Nc) is never encoded below 40 or above 120.  The decoder has measures
	; to reuse the previous Nc if the new one is outside that range.  But I suspect
	; that's for data loss handling (i.e. mobile), we won't have that issue here
	; so handling out of range won't be necessary.
	; ---------------------------------------------------------------------------
	; The xmaxc and xMc together make up the APCM inverse quantization, the result
	; of these are added to the dp as part of reconstructing the short term residual
	; signal.  The xmaxc is used to calculate the exponent and mantissa, of which
	; mantissa is used to collect the FAC (Normalised direct mantissa), multiplied
	; with xMc gets the ep to later be added to dp.
	;
	; Since there are only 8 possible xMc's and $40 possible xmaxc's, we can condense
	; these down to a LUT quite nicely, and eliminate a huge amount of calculations
	; including a mult_r.
	; ---------------------------------------------------------------------------
	; hl = Start of GSM frame:
	;
	;	NNNNNNNB
	;	BMMXXXXX
	;	X1112223
	; ---------------------------------------------------------------------------

	; --- loading all bits ---

		ld	a,(hl)					;  7	; load NNNNNNNB
		zand	011111110b				;  7	; keep NNNNNNN0 LTP lag (Nc)
		ld	b,a					;  4	; store NNNNNNN0
		xor	(hl)					;  7	; get 0000000B
		inc	hl					;  6	; next byte
		rlc	(hl)					; 15	; get lower bit of LTP gain (bc)
		adc	a					;  4	; fuse with upper bit
		zadd	QLB&0FFh				;  7	; set QLB table lower address
		ld	e,a					;  4	; ''
		ld	a,(hl)					;  7	; load BMMXXXXX
		inc	hl					;  6	; next byte
		rlc	(hl)					; 15	; get lowest bit of block amplitude (xmaxc)
		adc	a					;  4	; fuse with rest of XXXXX bits
		ld	d,a					;  4	; keep MMXXXXXX
		zand	11000000b				;  7	; get only the Mc bits MM000000
		ld	c,a					;  4	; store it away in c
		xor	d					;  4	; get 00XXXXXX
		ld	d,QLB>>8				;  7	; set QLB table upper address

	; --- setup xmaxc ---
	;  a = xmaxc
	;  c = Mc << 6
	;  b = Nc
	; de = QLB [bc]
	; -------------------

		rlca						;  4	; shift left 3 bits (to make way for xMc bits later)
		rlca						;  4	; ''
		rlca						;  4	; '' (Highest xmaxc bit is now in the carry
		ld	(xmaxcl+1),a				; 13	; save into or instruction for later/repeated use
		adc	a					;  4	; load the highest xmaxc bit out of the carry
		rlca						;  4	; align it
		zand	00000010b				;  7	; get only that bit
		zadd	epval>>8				;  7	; apply table address
		ld	(xmaxch+1),a				; 13	; save into ld instruction for later/repeated use

	rst	Flush

	; --- setup brp ---
	;  c = Mc << 6
	;  b = Nc
	; de = QLB [bc]
	; -----------------

		ld	a,(de)					;  7	; load lower byte of brp from QLB table
		ld	(brp+1),a				; 13	; save brp for later
		inc	d					;  4	; load higher byte of brp from QLB table
		ld	a,(de)					;  7	; ''
		ld	(brp+2),a				; 13	; save brp for later

	; --- setup xMc counter ---
	;  c = Mc << 6
	;  b = Nc
	; -------------------------

		ld	a,c					;  4	; get Mc
		rlca						;  4	; align to LSB
		rlca						;  4	; ''
		inc	a					;  4	; set so we detect 0 on decrement
		ld	(SkipxMc+1),a				; 13	; set counter

	; --- setup dp address ---
	;  b = Nc
	; ------------------------

		ld	de,dpTableList				; 10	; load dp table x40 multiply list (for each frame)
		ld	a,(frame+1)				; 13	; load frame slot number
		zadd	a					;  4	; x2 word table
		zadd	e					;  4	; advance to correct slot
		ld	e,a					;  4	; ''
		ex	de,hl					;  4	; swap for hl powers
		ld	a,(hl)					;  7	; load low byte dp address
		inc	l					;  4	; advance to high byte
		ld	h,(hl)					;  7	; load high byte dp address
		ld	l,a					;  4	; set low byte dp address
		; hl = dp [(frame * 40)]
		ld	c,b					;  4	; load Nc << 1 (already x2 from GSM data)
		xor	a					;  4	; clear carry
		ld	b,a					;  4	; clear b
		sbc	hl,bc					; 16	; go back to previous section of dp (i.e d [(frame * 40) - Nc])
		; hl = dp [(frame * 40) - Nc]
		ld	(dp+1),hl				; 16	; save "d [(frame * 40) - Nc]" into the loop
		ex	de,hl					;  4	; restore GSM data address

	; ---------------------------------------------------------------------------
	; Single sub-frame loop (40 samples per frame)
	; ---------------------------------------------------------------------------
	; This section deals with unpacking 40 samples out of the xMc data in this
	; frame, it handles the xMc unpacking and converting with xmaxc to create the
	; ep value, and sum this with the brp (bc) we collected earlier, multiplied
	; with the previous dp section 
	; ---------------------------------------------------------------------------

		ld	a,LoadxMc1&0FFh				;  7	; reset jp address of LoadxMc (at XX??)
		ld	(RetxMc+1),a				; 13	; ''
brp:		ld	bc,00000h				; 10	; load brp (QLB [bc])

	;	for (int i = 0; i <= 39; i++)

		ld	a,40-1					;  7	; reset sample counter
		ld	(dpsample+1),a				; 13	; ''
	rst	Flush

dpSampleLoop:

		; ---------------------------------------------------------------------------
		; We're unpacking 40 samples in a loop, however, only 13 of those times are
		; xMc's unpacked and summed, and they're done in increments of 3 (Mc + i*3)
		; so instead of two loops, we'll simply skip xMc unpacking 2 out of 3, and
		; we'll use Mc to dictate the starting point (see above loading Mc into SkipxMc+1)
		;
		; please note, by "sample" I don't mean final 100% result, I mean simply the
		; "dp" sample result (i.e. reconstructed short term residual signal)
		;
		; The next loop after at k_start to k_end will be the real sample loop
		; ---------------------------------------------------------------------------

		ld	de,00000h				; 10	; clear ep value (no ep value)
SkipxMc:	ld	a,000h					;  7	; load xMc counter
		dec	a					;  4	; decrease counter
		ld	(SkipxMc+1),a				; 13	; update counter
		jr	nz,NoxMc				;  7|12	; if no xMc this sample, skip loading xMc
		ld	a,3					;  7	; reset counter
		ld	(SkipxMc+1),a				; 13	; ''

		; ---------------------------------------------------------------------------
		; Unpacking xMc's into ep value (no ep table will be needed as we'll sum it
		; directly to the brp * dp section, but we'll refer to it as the ep value).
		; ---------------------------------------------------------------------------
		; hl = Somewhere in GSM xMc ("jp  LoadxMc1" reflects which one):
		;
		; X1112223
		; 33444555
		; 66677788
		; 8999AAAB
		; BBCCCDDD
		; ---------------------------------------------------------------------------
		;	for (int i = 0; i <= 12; i++)
		;	{
		;		short xMcur = (xMc [i] << 1) - 7;
		;		xMcur <<= 12;
		;		xMcur = (((int) fac * xMcur) + 0x4000) >> 0x0F;
		;		xMcur += xMadd;
		;		xMcur = xMcur >> expi;
		;		ep [Mc + (i * 3)] = xMcur;
		;	}
		; ---------------------------------------------------------------------------

RetxMc:		jp	LoadxMc1
xmaxcl:		zor	00000000b				;  7	; save XXXXX111
xmaxch:		ld	d,00000000b				;  7	; setup 000000X0
		ld	e,a					;  4	; de = 000000X0XXXXX111

		ld	a,(RetxMc+1)				; 13	; advance xMc routine
		zadd	LoadxMc2-LoadxMc1			;  7	; '' (every routine is D bytes in size)
		ld	(RetxMc+1),a				; 13	; ''

		ex	de,hl					;  4	; swap for hl powers
		ld	a,(hl)					;  7	; load lower byte of ep value
		inc	h					;  4	; advance to upper byte
		ld	h,(hl)					;  7	; load higher byte of ep value
		ld	l,a					;  4	; ''
		ex	de,hl					;  4	; restore hl (de = value)
	rst	Flush

NoxMc:
		push	hl					; 11	; store GSM data address

		; ---------------------------------------------------------------------------
		; Unpacking xMc's into ep value (no ep table will be needed as we'll sum it
		; directly to the brp * dp section, but we'll refer to it as the ep value).
		; ---------------------------------------------------------------------------
		; de = ep value to be added to brp * dp
		; b  = Nc << 1 (Nr)
		; ---------------------------------------------------------------------------
		;	for (int k = 0; k <= 39; k++)
		;	{
		;		short drpp = (((int) brp * dp [(frame * 40) - Nr + k]) + 0x4000) >> 0x0F;
		;		drpp += ep [k];
		;		dp [(frame * 40) + k] = drpp;	// write 160 200 240 280
		;		df [(frame * 40) + k] = drpp;	// write   0  40  80 120
		;	}
		; ---------------------------------------------------------------------------

dp:		ld	hl,00000h				; 10	; load dp table address (with - Nc)
		push	de					; 11	; store ep value
		ld	e,(hl)					;  7	; load lower ep byte
		inc	l					;  4	; load upper ep byte
		ld	d,(hl)					;  7	; ''
		inc	hl					;  6	; next word address
		ld	(dp+1),hl				; 16	; update dp table address
		ex	de,hl					;  4	; load ep value into hl for mult_r
		rst	mult_r						; multiply brp (in bc) with ep value (in hl) to create drpp
		pop	de					; 10	; restore ep value
		zadd	hl,de					; 11	; add ep to drpp

		ld	(ix+0),l				; 19	; write   0  40  80 120
		ld	(iy+0),l				; 19	; write 160 200 240 280
		inc	ixl					;  8	; advance to higher byte
		inc	iyl					;  8	; ''
		ld	(ix+0),h				; 19	; write   0  40  80 120
		ld	(iy+0),h				; 19	; write 160 200 240 280
		inc	ix					; 10	; advance to next k
		inc	iy					; 10	; ''

		; ---------------------------------------------------------------------------
		; loop back for next dp sample...
		; ---------------------------------------------------------------------------

		pop	hl					; 10	; restore GSM data address

		; need to loop back for all 40 samples

dpsample:	ld	a,000h					;  7	; load dp sample counter
		dec	a					;  4	; decrease counter
		ld	(dpsample+1),a				; 13	; update counter
		jp	p,dpSampleLoop				; 10	; loop back if we've not done all 40 sample (i.e. 39..0)

	; ---------------------------------------------------------------------------
	; convert the LARpp to rrp here
	; ---------------------------------------------------------------------------
	; a = current frame number 3, 2, 1, 0
	; ---------------------------------------------------------------------------

		push	hl					; 11	; store GSM data address
		ld	hl,LARcur				; 10	; load current/previous LARpp pointers
		ld	d,h					;  4	; set d to LARpp area too
		ld	e,(hl)					;  7	; load LARpp(j) (current)
		inc	l					;  4	; load LARpp(j-1) (previous)
		ld	l,(hl)					;  7	; ''

		; de = LAR''(j)
		; hl = LAR''(j-1)

		ld	a,(frame+1)				; 13	; load frame number
		dec	a					;  4	; setup frame number ready for condition testing
		cp	1					;  7	; set flags, S, C, and Z
		jp	c,LAR_3					; 10	; if it was 01, branch for frame 3 (00 to FF, carry set)
		jp	m,LAR_4					; 10	; if it was 00, branch for frame 4 (FF to FE, negative set, no carry)
		jp	z,LAR_2					; 10	; if it was 02, branch for frame 2 (01 to 00, zero set, no carry, no negative)
									; if it was 03, keep here for frame 1 (02 to 01, none of the above)
		; -------------------------------------------------------
		; Interpolation of LARpp (1st coefficient set (0 to 12))
		;	for (int i = 1; i <= 8; i++)
		;	{
		;		short LAR = (LARpp [pre] [i] >> 2) + (LARpp [cur] [i] >> 2) + (LARpp [pre] [i] >> 1);
		;		rrp [i] = GSM_ReflectLAR (LAR);
		;	}
		;	k_start = 0;
		;	k_end = 12;
		; -------------------------------------------------------

LAR_1:

		; --- LAR''(j-1) >> 1 ---
		; All 4 frames of LAR reflection use the following:
		;   LAR''(j) >> 2     LAR''(j) >> 1     LAR''(j)
		;   LAR''(j-1) >> 2   LAR''(j-1) >> 1
		; Since LAR''(j-1) (previous) is never used in any of the
		; 4 frames, we'll shift it right once here, and store it back
		; again, to save doing shift right >> 2 for the other frames
		; -----------------------

		ld	a,rrp&0FFh				;  4	; reset i counter

	.Nexti:
	bit	1,a
	jr	nz,.nofl
	rst	Flush
	.nofl:
		ld	c,(hl)					;  7	; load LAR''(j-1) value
		inc	l					;  4	; ''
		ld	b,(hl)					;  7	; ''
		dec	l					;  4	; go back to resave >> 1
		sra	b					;  8	; shift right by 1
		rr	c					;  8	; ''
		ld	(hl),c					;  7	; store LAR''(j-1) as >> 1
		inc	l					;  4	; ''
		ld	(hl),b					;  7	; ''
		inc	l					;  4	; ''
		push	hl					; 11	; store LAR''(j-1) table

		; bc = LAR''(j-1) >> 1

		ld	l,c					;  4	; store >> 1
		ld	h,b					;  4	; ''
		sra	b					;  8	; shift right to >> 2
		rr	c					;  8	; ''
		zadd	hl,bc					; 11	; sum LAR''(j-1) >> 1 and LAR''(j-1) >> 2

		; hl = (LAR''(j-1) >> 1) + (LAR''(j-1) >> 2)

		ex	de,hl					;  4	; get LAR''(j)
		ld	c,(hl)					;  7	; ''
		inc	l					;  4	; ''
		ld	b,(hl)					;  7	; ''
		inc	l					;  4	; ''
		ex	de,hl					;  4	; restore result
		sra	b					;  8	; shift right x1
		rr	c					;  8	; ''
		sra	b					;  8	; shift right x2
		rr	c					;  8	; ''

		; bc = LAR''(j) >> 2

		zadd	hl,bc					; 11	; hl = (LARpp(j-1) >> 2) + (LARpp(j-1) >> 1) + (LARpp(j) >> 2)
		push	de					; 11	; store LAR''(j) table
		rst	ReflEndLAR				; 11	; perform reflection LAR, save result, and check for end
		pop	de					; 10	; restore LAR''(j) table
		pop	hl					; 10	; restore LAR''(j-1) table
		jp	nz,.Nexti				; 10	; if not finished reflection, loop

		ld	hl,dpTable+(0*2)			; 10	; dp [k_start]
		ld	a,(dpTable+((12+1)*2))&0FFh		;  7	; k_end
		jp	LAR_Done				; 10	; finish

		; -------------------------------------------------------
		; Interpolation of LARpp (2nd coefficient set (13 to 26))
		;	for (int i = 1; i <= 8; i++)
		;	{
		;		short LAR = (LARpp [pre] [i] >> 1) + (LARpp [cur] [i] >> 1);
		;		rrp [i] = GSM_ReflectLAR (LAR);
		;	}
		;	k_start = 13;
		;	k_end = 26;
		; -------------------------------------------------------

LAR_2:

		; --- LAR''(j-1) >> 1 ---
		; Already shifted right once on first frame
		; -----------------------

		ld	a,rrp&0FFh				;  4	; reset i counter

	.Nexti:
	bit	1,a
	jr	nz,.nofl
	rst	Flush
	.nofl:
		ld	c,(hl)					;  7	; load LAR''(j-1) >> 1 value
		inc	l					;  4	; ''
		ld	b,(hl)					;  7	; ''
		inc	l					;  4	; ''
		push	hl					; 11	; store LAR''(j-1) table

		; bc = LAR''(j-1) >> 1

		ex	de,hl					;  4	; get LAR''(j)
		ld	e,(hl)					;  7	; ''
		inc	l					;  4	; ''
		ld	d,(hl)					;  7	; ''
		inc	l					;  4	; ''
		ex	de,hl					;  4	; restore result
		sra	h					;  8	; shift right x1
		rr	l					;  8	; ''

		; hl = LAR''(j) >> 1

		zadd	hl,bc					; 11	; hl = (LARpp(j-1) >> 1) + (LARpp(j) >> 2)
		push	de					; 11	; store LAR''(j) table
		rst	ReflEndLAR				; 11	; perform reflection LAR, save result, and check for end
		pop	de					; 10	; restore LAR''(j) table
		pop	hl					; 10	; restore LAR''(j-1) table
		jp	nz,.Nexti				; 10	; if not finished reflection, loop

		ld	hl,dpTable+(13*2)			; 10	; dp [k_start]
		ld	a,(dpTable+((26+1)*2))&0FFh		;  7	; k_end
		jp	LAR_Done				; 10	; finish

		; -------------------------------------------------------
		; Interpolation of LARpp (3rd coefficient set (27 to 39))
		;	for (int i = 1; i <= 8; i++)
		;	{
		;		short LAR = (LARpp [pre] [i] >> 2) + (LARpp [cur] [i] >> 2) + (LARpp [cur] [i] >> 1);
		;		rrp [i] = GSM_ReflectLAR (LAR);
		;	}
		;	k_start = 27;
		;	k_end = 39;
		; -------------------------------------------------------

LAR_3:

		; --- LAR''(j-1) >> 1 ---
		; Already shifted right once on first frame
		; -----------------------

		ld	a,rrp&0FFh				;  4	; reset i counter
		ex	de,hl					;  4	; same as frame 1, except cur and prev are swapped

	.Nexti:
	bit	1,a
	jr	nz,.nofl
	rst	Flush
	.nofl:
		ld	c,(hl)					;  7	; load LAR''(j) value
		inc	l					;  4	; ''
		ld	b,(hl)					;  7	; ''
		inc	l					;  4	; ''
		push	hl					; 11	; store LAR''(j) table
		sra	b					;  8	; shift right to >> 1
		rr	c					;  8	; ''

		; bc = LAR''(j) >> 1

		ld	l,c					;  4	; store >> 1
		ld	h,b					;  4	; ''
		sra	b					;  8	; shift right to >> 2
		rr	c					;  8	; ''
		zadd	hl,bc					; 11	; sum LAR''(j) >> 1 and LAR''(j) >> 2

		; hl = (LAR''(j) >> 1) + (LAR''(j) >> 2)

		ex	de,hl					;  4	; get LAR''(j-1)
		ld	c,(hl)					;  7	; ''
		inc	l					;  4	; ''
		ld	b,(hl)					;  7	; ''
		inc	l					;  4	; ''
		ex	de,hl					;  4	; restore result
		sra	b					;  8	; shift right x2
		rr	c					;  8	; ''

		; bc = LAR''(j) >> 2

		zadd	hl,bc					; 11	; hl = (LARpp(j) >> 2) + (LARpp(j) >> 1) + (LARpp(j-1) >> 2)
		push	de					; 11	; store LAR''(j-1) table
		rst	ReflEndLAR				; 11	; perform reflection LAR, save result, and check for end
		pop	de					; 10	; restore LAR''(j-1) table
		pop	hl					; 10	; restore LAR''(j) table
		jp	nz,.Nexti				; 10	; if not finished reflection, loop

		ld	hl,dpTable+(27*2)			; 10	; dp [k_start]
		ld	a,(dpTable+((39+1)*2))&0FFh		;  7	; k_end
		jp	LAR_Done				; 10	; finish

		; -------------------------------------------------------
		; Interpolation of LARpp (4th coefficient set (40 to 159))
		;	for (int i = 1; i <= 8; i++)
		;	{
		;		short LAR = LARpp [cur] [i];
		;		rrp [i] = GSM_ReflectLAR (LAR);
		;	}
		;	k_start = 40;
		;	k_end = 159;
		; -------------------------------------------------------

LAR_4:

		ld	a,rrp&0FFh				;  4	; reset i counter
		ex	de,hl					;  4	; access LARpp(j) only...

	.Nexti:
	bit	1,a
	jr	nz,.nofl
	rst	Flush
	.nofl:
		ld	e,(hl)					;  7	; load LAR''(j)
		inc	l					;  4	; ''
		ld	d,(hl)					;  7	; ''
		inc	l					;  4	; ''
		push	hl					; 11	; store LAR''(j) table
		ex	de,hl					;  4	; store in hl
		rst	ReflEndLAR				; 11	; perform reflection LAR, save result, and check for end
		pop	hl					; 10	; restore LAR''(j) table
		jp	nz,.Nexti				; 10	; if not finished reflection, loop

		ld	hl,dpTable+(40*2)			; 10	; dp [k_start]
		ld	a,(dpTable+((159+1)*2))&0FFh		;  7	; k_end

LAR_Done:

	; ---------------------------------------------------------------------------
	; Single sub-frame loop (k_start to k_end samples per frame)
	; ---------------------------------------------------------------------------
	; This section deals with converting the dp samples into real samples
	; frame 1 =  0 to 12
	; frame 2 = 13 to 26
	; frame 3 = 27 to 39
	; frame 4 = 40 to 159
	; ---------------------------------------------------------------------------

	;	for (int k = k_start; k <= k_end; k++)

		; hl = dp [k_start]
		; a = k_end+1

SampleLoop:
	bit	1,l
	jr	nz,.nofl
	rst	Flush
	.nofl:
		ld	e,(hl)					;  7	; load dp [k] sample/signal
		inc	l					;  4	; ''
		ld	d,(hl)					;  7	; ''
		inc	hl					;  6	; ''
		ld	(.Samp+1),de				; 20	; store sample in the loop ready for use...
		push	hl					; 11	; store dp [k] table
		push	af					; 11	; store k_end+1

		; ---------------------------------------------------------------------------
		; STS filtering, Deemphasis filtering, Upscale, Truncate, and Output
		; ---------------------------------------------------------------------------
		; This converts the "dp" sample results (i.e. reconstructed short term residual signal)
		; into the final 13-bit actual PCM sample, ready for playback
		; ---------------------------------------------------------------------------
		;	for (int k = k_start; k <= k_end; k++)
		;	{
		;		// Short term synthesis filtering section
		;		short Sample = dp [k];	 // always use the first 40 (first frame) for all 4 frames
		;		for (int i = 1; i <= 8; i++)
		;		{
		;			Sample -= (((int) rrp [9-i] * v [8-i]) + 0x4000) >> 0x0F;
		;			v [9-i] = v [8-i] + ((((int) rrp [9-i] * Sample) + 0x4000) >> 0x0F);
		;		}
		;		v [0] = Sample;
		;		// Deemphasis filtering
		;		Sample += (((int) msr * 28180) + 0x4000) >> 0x0F;
		;		msr = Sample;
		;		// Upscaling of the output signal
		;		Sample = GSM_add (Sample, Sample);	// This is the ONLY one which seems to need a cap (for obvious reasons).
		;		// Truncation of the output variable
		;		Sample &= 0xFFF8;
		;		// Output of the sample...
		;		Samples [SampPos++] = (float) Sample;
		;	}
		; ---------------------------------------------------------------------------

		; Short term synthesis filtering section

		ld	hl,v+(8*2)				; 10	; load v/rrp table at last word on low byte

	.STS_Filter:
		dec	l					;  4	; move to higher byte
		ld	b,(hl)					;  7	; load upper byte v
		inc	h					;  4	; go to rrp table
		ld	d,(hl)					;  7	; load upper byte rrp
		dec	l					;  4	; move to lower byte
		ld	e,(hl)					;  7	; load lower byte rrp
		dec	h					;  4	; go to v table again
		ld	c,(hl)					;  7	; load lower byte v
		push	hl					; 11	; store v/rrp table

		; de = rrp [9-i]
		; bc = v [8-i]

		push	bc					; 11	; store v [8-i]	(reuse later)
		push	de					; 11	; store rrp [9-i] (reuse later)
		ex	de,hl					;  4	; hl * bc
		rst	mult_r						; multiply and round
		ex	de,hl					;  4	; de = mult_r (rrp [9-i], v [8-i])
	.Samp:	ld	hl,00000h				; 10	; load sample
		sbc	hl,de					; 16	; Sample -= de (carry cleared by mult_r)
		ld	(.Samp+1),hl				; 16	; update sample

		pop	bc					; 10	; restore rrp [9-i]
		rst	mult_r						; multiply and round
		pop	bc					; 10	; restore v [8-i]
		zadd	hl,bc					; 11	; v [8-i] + mult_r (sample * rrp [9-i])
		ex	de,hl					;  4	; de = result

		pop	hl					; 10	; restore v/rrp table
		ld	a,l					;  4	; store [i] position
		inc	l					;  4	; advance [i] position from v [8-i] to v [9-i]
		inc	l					;  4	; ''
		ld	(hl),e					;  7	; save result
		inc	l					;  4	; ''
		ld	(hl),d					;  7	; ''
		ld	l,a					;  4	; restore [i] position

		ld	a,v&0FFh				;  7	; check for beginning of v table
		cp	l					;  4	; ''
		jp	nz,.STS_Filter				; 10	; if not reached v [0], repeat

		ld	de,(.Samp+1)				; 20	; load sample
		ld	(hl),e					;  7	; save sample to v [0]
		inc	l					;  4	; ''
		ld	(hl),d					;  7	; ''

		; Deemphasis filtering

	.msr:	ld	bc,00000h				; 10	; load msr

		; --- Real multiplication with 6E14 ---
		; The code block below was an attempt at multiplying
		; the fixed 6E14 using lookup tables, moderately
		; successful, though not perfect.
		; -------------------------------------

		ld	hl,06E14h
		push	de
		rst	mult_r
		pop	de

		; --- hl = GSM_mult_r (msr, 6E14) ---
		; Old table swap method, won't work because of rounding...
		; -----------------------------------
	;	ld	h,MULTR6E14>>8				;  7	; prepare low mult low result
	;	bit	7,b					;  8	; is the msr negative?
	;	jr	nz,.Neg					;  7|12	; if so, jump to negation version
	;	ld	l,c					;  4	; set low mult
	;	ld	c,(hl)					;  7	; load low mult low result
	;	inc	h					;  4	; advance to high mult low result
	;	ld	l,b					;  4	; set high mult
	;	ld	b,000h					;  7	; bc is positive
	;	jp	.Pos					; 10	; continue with upper byte multiplication
	;
	;.Neg:
	;	xor	a					;  4	; negate low byte msr
	;	zsub	c					;  4	; ''
	;	ld	l,a					;  4	; load result
	;	xor	a					;  4	; '' ...as negative
	;	zsub	(hl)					;  7	; ''
	;	inc	h					;  4	; setup hl for high byte multiplication
	;	ld	l,b					;  4	; ''
	;	ld	c,a					;  4	; store low byte multiplication word
	;	ld	b,0FFh					;  7	; '' ...as negative
	;inc	l	; advance l (negative low byte carries up)
	;
	;.Pos:
	;	push	bc					; 11	; store low mult multiplcation
	;	ld	c,(hl)					;  7	; load high mult low result
	;	inc	h					;  4	; advance to high result
	;	ld	b,(hl)					;  7	; load high mult high result
	;	pop	hl					; 10	; reload low mult multiplication
	;	zadd	hl,bc					; 11	; add high mult multiplication
		; -----------------------------------

		zadd	hl,de					; 11	; add sample to msr
		ld	(.msr+1),hl				; 16	; update msr

		; Upscaling of the output signal
		zadd	hl,hl					; 11	; scale signal back up x2

		; Truncation of the output variable
		ld	a,h					;  4	; load upper 8-bits of 13-bit signal
		zadd	080h					;  7	; convert to unsigned

		; Output of the sample...

Buffer:		ld	hl,Output				; 10	; load buffer address
		ld	(hl),a					;  7	; save sample
		inc	l					;  4	; advance to next sample
		ld	(Buffer+1),hl				; 16	; update sample address

		; ---------------------------------------------------------------------------
		; loop back for next sample...
		; ---------------------------------------------------------------------------

		pop	af					; 10	; restore k_end+1
		pop	hl					; 10	; restore dp [k] table
		cp	l					;  4	; check if k has reached k_end
		jp	nz,SampleLoop				; 10	; if not, loop

		pop	hl					; 10	; restore GSM data address

	; ---------------------------------------------------------------------------
	; loop back for next frame...
	; ---------------------------------------------------------------------------

frame:		ld	a,000h					;  7	; load frame counter
		dec	a					;  4	; decrease counter
		ld	(frame+1),a				; 13	; update counter
		jp	p,FrameLoop				; 10	; loop back if we've not done all 4 frames (i.e. 3, 2, 1, 0)

	; ---------------------------------------------------------------------------
	; End of GSM block
	; ---------------------------------------------------------------------------

		; Swap LARpp buffers

		ex	de,hl					;  4	; store GSM data address
		ld	hl,LARcur				; 10	; load current/previous LARpp pointers
		ld	a,(hl)					;  7	; load LARpp(j) current
		inc	l					;  4	; load LARpp(j-1) previous
		ld	b,(hl)					;  7	; ''
		ld	(hl),a					;  7	; previous is now current
		dec	l					;  4	; current is now previous
		ld	(hl),b					;  7	; ''
		ex	de,hl					;  4	; restore GSM data address

		jp	DecodeBlock				; 10	; loop back and decode next block

; ===========================================================================
; ---------------------------------------------------------------------------
; Since generating the LARpp(j) out of the LARc doesn't require a previous
; frame/block, its generation is predictable and limited.  Thus a simplified
; swap table is very usable.  Here are all possible outcomes:
;
;	LARc[1]	LARc[2]	LARc[3]	LARc[4]	LARc[5]	LARc[6]	LARc[7]	LARc[8]
; 
;000000	999A	999A	C000	DCCC	D998	ECCC	E666	F332
;000001	9CCE	9CCE	C334	E000	DE4A	F112	EE14	FA72
;000010	A000	A000	C666	E334	E2FA	F556	F5C2	01B4
;000011	A334	A334	C99A	E666	E7AC	F99A	FD70	08F4
;000100	A666	A666	CCCE	E99A	EC5E	FDDE	051E	1034
;000101	A99A	A99A	D000	ECCC	F110	0222	0CCC	1776
;000110	ACCE	ACCE	D334	F000	F5C0	0666	147A	1EB6
;000111	B000	B000	D666	F334	FA72	0AAA	1C26	25F6
;001000	B334	B334	D99A	F666	FF24	0EEE
;001001	B666	B666	DCCC	F99A	03D4	1334
;001010	B99A	B99A	E000	FCCC	0886	1778
;001011	BCCE	BCCE	E334	0000	0D38	1BBC
;001100	C000	C000	E666	0334	11EA	2000
;001101	C334	C334	E99A	0666	169A	2444
;001110	C666	C666	ECCC	099A	1B4C	2888
;001111	C99A	C99A	F000	0CCC	1FFE	2CCC
;010000	CCCE	CCCE	F334	1000
;010001	D000	D000	F666	1334
;010010	D334	D334	F99A	1666
;010011	D666	D666	FCCC	199A
;010100	D99A	D99A	0000	1CCC
;010101	DCCC	DCCC	0334	2000
;010110	E000	E000	0666	2334
;010111	E334	E334	099A	2666
;011000	E666	E666	0CCC	299A
;011001	E99A	E99A	1000	2CCC
;011010	ECCC	ECCC	1334	3000
;011011	F000	F000	1666	3334
;011100	F334	F334	199A	3666
;011101	F666	F666	1CCC	399A
;011110	F99A	F99A	2000	3CCC
;011111	FCCC	FCCC	2334	4000
;100000	0000	0000
;100001	0334	0334
;100010	0666	0666
;100011	099A	099A
;100100	0CCC	0CCC
;100101	1000	1000
;100110	1334	1334
;100111	1666	1666
;101000	199A	199A
;101001	1CCC	1CCC
;101010	2000	2000
;101011	2334	2334
;101100	2666	2666
;101101	299A	299A
;101110	2CCC	2CCC
;101111	3000	3000
;110000	3334	3334
;110001	3666	3666
;110010	399A	399A
;110011	3CCC	3CCC
;110100	4000	4000
;110101	4332	4332
;110110	4666	4666
;110111	499A	499A
;111000	4CCC	4CCC
;111001	5000	5000
;111010	5332	5332
;111011	5666	5666
;111100	599A	599A
;111101	5CCC	5CCC
;111110	6000	6000
;111111	6332	6332
; ---------------------------------------------------------------------------
; These are the results of LARpp(1..8) after calculating from LARc
; First 100 are the low byte, second 100 are the high byte
; the low byte offset is dictated by the LARcTables which have an "Add"
; parameter for the macro
; ---------------------------------------------------------------------------
; The LARpp(j) and LARpp(j-1) results table is stored neatly at the end of
; the low byte swap table (it also keeps them all at the same upper address)
; ---------------------------------------------------------------------------

		align	0100h

LARpp1n2:	db	09Ah,0CEh,000h,034h,066h,09Ah,0CEh,000h,034h,066h,09Ah,0CEh,000h,034h,066h,09Ah
		db	0CEh,000h,034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh
		db	000h,034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh,000h
		db	034h,066h,09Ah,0CCh,000h,032h,066h,09Ah,0CCh,000h,032h,066h,09Ah,0CCh,000h,032h

LARpp3:		db	000h,034h,066h,09Ah,0CEh,000h,034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh,000h
		db	034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh,000h,034h

LARpp4:		db	0CCh,000h,034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh
		db	000h,034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh,000h,034h,066h,09Ah,0CCh,000h

LARpp5:		db	098h,04Ah,0FAh,0ACh,05Eh,010h,0C0h,072h,024h,0D4h,086h,038h,0EAh,09Ah,04Ch,0FEh

LARpp6:		db	0CCh,012h,056h,09Ah,0DEh,022h,066h,0AAh,0EEh,034h,078h,0BCh,000h,044h,088h,0CCh

LARpp7:		db	066h,014h,0C2h,070h,01Eh,0CCh,07Ah,026h

LARpp8:		db	032h,072h,0B4h,0F4h,034h,076h,0B6h,0F6h

QLB:		db	0CDh, 0CDh, 033h, 0FFh

		;	v[8-8] v[8-7]] v[8-6] v[8-5] v[8-4] v[8-3] v[8-2] v[8-1] v[9-1]
v:		dw	00000h,00000h,00000h,00000h,00000h,00000h,00000h,00000h, 00000h	; rrp is on opposite of v

LARcur:		db	(LARppj&0FFh)	; low byte where LAR''(j) current is
LARpre:		db	(LARppjm&0FFh)	; low byte where LAR''(j-1) previous is
LARppj:		dsb	8*2,0		; LAR''(j) (a.k.a. LARpp current)
LARppjm:	dsb	8*2,0		; LAR''(j-1) (a.k.a. LARpp previous)

dpTableList:	dw	dpTable+(120*2)		; frame 4's dp
		dw	dpTable+(80*2)		; frame 3's dp
		dw	dpTable+(40*2)		; frame 2's dp
		dw	dpTable+(0*2)		; frame 1's dp

		; $10 bytes left...

		align	0100h

		db	099h,09Ch,0A0h,0A3h,0A6h,0A9h,0ACh,0B0h,0B3h,0B6h,0B9h,0BCh,0C0h,0C3h,0C6h,0C9h
		db	0CCh,0D0h,0D3h,0D6h,0D9h,0DCh,0E0h,0E3h,0E6h,0E9h,0ECh,0F0h,0F3h,0F6h,0F9h,0FCh
		db	000h,003h,006h,009h,00Ch,010h,013h,016h,019h,01Ch,020h,023h,026h,029h,02Ch,030h
		db	033h,036h,039h,03Ch,040h,043h,046h,049h,04Ch,050h,053h,056h,059h,05Ch,060h,063h

		db	0C0h,0C3h,0C6h,0C9h,0CCh,0D0h,0D3h,0D6h,0D9h,0DCh,0E0h,0E3h,0E6h,0E9h,0ECh,0F0h
		db	0F3h,0F6h,0F9h,0FCh,000h,003h,006h,009h,00Ch,010h,013h,016h,019h,01Ch,020h,023h

		db	0DCh,0E0h,0E3h,0E6h,0E9h,0ECh,0F0h,0F3h,0F6h,0F9h,0FCh,000h,003h,006h,009h,00Ch
		db	010h,013h,016h,019h,01Ch,020h,023h,026h,029h,02Ch,030h,033h,036h,039h,03Ch,040h

		db	0D9h,0DEh,0E2h,0E7h,0ECh,0F1h,0F5h,0FAh,0FFh,003h,008h,00Dh,011h,016h,01Bh,01Fh

		db	0ECh,0F1h,0F5h,0F9h,0FDh,002h,006h,00Ah,00Eh,013h,017h,01Bh,020h,024h,028h,02Ch

		db	0E6h,0EEh,0F5h,0FDh,005h,00Ch,014h,01Ch

		db	0F3h,0FAh,001h,008h,010h,017h,01Eh,025h

		db	00Ch, 02Ch, 053h, 07Fh

		;	rrp[1] rrp[2] rrp[3] rrp[4] rrp[5] rrp[6] rrp[7] rrp[8]
rrp:		dw	00000h,00000h,00000h,00000h,00000h,00000h,00000h,00000h	; v is on opposite of rrp

; ---------------------------------------------------------------------------
; dp buffer (with underflow)
; ---------------------------------------------------------------------------

dfTable:	dsb	40*2,0
		dsb	40*2,0
		dsb	40*2,0
		dsb	40*2,0
dpTable:	dsb	40*2,0
		dsb	40*2,0
		dsb	40*2,0
		dsb	40*2,0

; ---------------------------------------------------------------------------
; xMc load routines (one for each 3 bits)
; ---------------------------------------------------------------------------
; alignment just needs to be such that all LoadxMc's are in the same
; 0100h block, only the low byte is added to, this fits perfectly after the
; dp tables, and they all remain in the same 0100h block space
; ---------------------------------------------------------------------------

LoadxMc1:	; 31
		ld	d,ASR4M00000111>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load X1112223
		ld	a,(de)					;  7	; load 00000111
		jp	RetxMc+3				; 10	; return back

		dsb	7,0FFh

LoadxMc2:	; 31
		ld	d,ASR1M00000111>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load X1112223
		ld	a,(de)					;  7	; load 00000222
		jp	RetxMc+3				; 10	; return back

		dsb	7,0FFh

LoadxMc3:	; 63
		ld	d,ASL2M00000100>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load X1112223
		ld	a,(de)					;  7	; load 00000300
		inc	d					;  4	; next table
		inc	hl					;  6	; next byte
		ld	e,(hl)					;  7	; load 33444555
		ex	de,hl					;  4	; load 00000033
		zor	(hl)					;  7	; ''
		ex	de,hl					;  4	; ''
		jp	RetxMc+3				; 10	; return back

		dsb	1,0FFh

LoadxMc4:	; 31
		ld	d,ASR3M00000111>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load 33444555
		ld	a,(de)					;  7	; load 00000444
		jp	RetxMc+3				; 10	; return back

		dsb	7,0FFh

LoadxMc5:	; 30
		ld	a,(hl)					;  7	; load 33444555
		zand	00000111b				;  7	; load 00000555
		inc	hl					;  6	; next byte
		jp	RetxMc+3				; 10	; return back

		dsb	7,0FFh

LoadxMc6:	; 31
		ld	d,ASR5M00000111>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load 66677788
		ld	a,(de)					;  7	; load 00000666
		jp	RetxMc+3				; 10	; return back

		dsb	7,0FFh

LoadxMc7:	; 31
		ld	d,ASR2M00000111>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load 66677788
		ld	a,(de)					;  7	; load 00000777
		jp	RetxMc+3				; 10	; return back

		dsb	7,0FFh

LoadxMc8:	; 63
		ld	d,ASL1M00000110>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load 66677788
		inc	hl					;  6	; next byte
		ld	a,(hl)					;  7	; load 8999AAAB
		rlca						;  4	; rotate to 999AAAB8
		zand	00000001b				;  7	; get only 00000008
		ex	de,hl					;  4	; swap for hl powers
		zor	(hl)					;  7	; load 00000880
		ex	de,hl					;  4	; restore GSM data
	; Old table version (ended up being the same number of cycles... just 1 byte smaller)
	;	ld	d,ASL1M00000110>>8			;  7	; setup table for bit swap
	;	ld	e,(hl)					;  7	; load 66677788
	;	ld	a,(de)					;  7	; load 00000880
	;	inc	d					;  4	; next table
	;	inc	hl					;  6	; next byte
	;	ld	e,(hl)					;  7	; load 8999AAAB
	;	ex	de,hl					;  4	; load 00000008
	;	zor	(hl)					;  7	; ''
	;	ex	de,hl					;  4	; ''
		jp	RetxMc+3				; 10	; return back

		dsb	0,0FFh

LoadxMc9:	; 31
		ld	d,ASR4M00000111>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load 8999AAAB
		ld	a,(de)					;  7	; load 00000999
		jp	RetxMc+3				; 10	; return back

		dsb	7,0FFh

LoadxMcA:	; 31
		ld	d,ASR1M00000111>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load 8999AAAB
		ld	a,(de)					;  7	; load 00000AAA
		jp	RetxMc+3				; 10	; return back

		dsb	7,0FFh

LoadxMcB:	; 63
		ld	d,ASL2M00000100>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load 8999AAAB
		ld	a,(de)					;  7	; load 00000B00
		inc	d					;  4	; next table
		inc	hl					;  6	; next byte
		ld	e,(hl)					;  7	; load BBCCCDDD
		ex	de,hl					;  4	; load 000000BB
		zor	(hl)					;  7	; ''
		ex	de,hl					;  4	; ''
		jp	RetxMc+3				; 10	; return back

		dsb	1,0FFh

LoadxMcC:	; 31
		ld	d,ASR3M00000111>>8			;  7	; setup table for bit swap
		ld	e,(hl)					;  7	; load BBCCCDDD
		ld	a,(de)					;  7	; load 00000CCC
		jp	RetxMc+3				; 10	; return back

		dsb	7,0FFh

LoadxMcD:	; 47

		xor	a					;  4	; prevent a 14th set from being read (1 to 13 only).
		ld	(SkipxMc+1),a				; 13	; '' (Please see note below)

		ld	a,(hl)					;  7	; load BBCCCDDD
		zand	00000111b				;  7	; load 00000DDD
		inc	hl					;  6	; next byte
		jp	RetxMc+3				; 10	; return back

	; --- Note of SkipxMc clearing ---
	; The original ep buffer of 40 slots (0 to 39), would have the 13 xMc
	; coefficients written into every 3rd slot (relative to Mc's start point),
	; while the rest of the ep buffer would be cleared.
	;
	; if Mc were 2, then ep 2, 5, 8, 11, 14, 17, etc... would have the
	; xMc 1, 2, 3, 4, 5, 6, etc... written.
	;
	; So... the loop would start as:
	;  Clear 0
	;  Clear 1
	;  Write 2	xMc 1
	;  Clear 3
	;  Clear 4
	;  Write 5	xMc 2
	;  Clear 6
	;  etc...
	;
	; Since Mc can be 0, 1, 2, 3, the first xMc can be written to ep slot
	; 0, 1, 2, or 3, and the repeating pattern of every 3 slots continues
	; from there.
	;
	; Mc 0 is a unique case, because the 13th xMc bit gets written to
	; slot 36, and since there are 3 slots to go, a 14th one accidentally
	; gets written to 39.
	;
	; Example end of Mc = 0:
	;
	;  etc...
	;  Clear 32
	;  Write 33	xMc 12
	;  Clear 34
	;  Clear 35
	;  Write 36	xMc 13 (last one)
	;  Clear 37
	;  Clear 38
	;  Write 39	xMc 14 !whoops!
	;
	; By clearing the "SkipxMc" counter on the 13th write, it'll underflow
	; to FF on the next "dec a", and so it'll never trigger again, thus ensuring
	; slot 39 stays clear.
	; --------------------------------

; ===========================================================================
; ---------------------------------------------------------------------------
; Output sample area
; ---------------------------------------------------------------------------
; 2 buffers are normally used, though since decoding 2G data is so slow, samples
; are flushed out likely way in advance, before we can even write, so I
; suspect we'll only need 1 buffer.
; ---------------------------------------------------------------------------

	; use space here for code (not much left)

		align	0100h
Output:		dsb	160,080h

	; use space here for code (not much left)

; ===========================================================================
; ---------------------------------------------------------------------------
; Computation of the rp[1..8] from the interpolated LARp[1..8]
; ---------------------------------------------------------------------------
; The input of this procedure is the interpolated LARp[1..8] array.
; The reflection coefficients, rp[i], are used in the analysis
; filter and in the synthesis filter.
; ---------------------------------------------------------------------------
; In: hl = LAR | Out: hl = rp
; ---------------------------------------------------------------------------

ReflectLAR:
		ex	de,hl					;  4	; load rp to hl
		bit	7,d					;  8	; is rp (LAR) negative?
		jr	nz,.Neg					;  7|12	; if so, branch
		ld	hl,-11059				; 10	; check if lower than 11059
		zadd	hl,de					; 11	; ''
		jr	c,.No11059				;  7|12	; if (rp >= 11059)
		ex	de,hl					;  4	; rp <<= 1
		zadd	hl,hl					; 11	; 11059/32767 = 0.3375 = 0.675/2
		jp	(ReflectRET+3)				; 10	; done

	.No11059:
		ld	hl,-20070				; 10	; check if lower than 20070
		zadd	hl,de					; 11	; ''
		jr	c,.No20070				;  7|12	; if (rp >= 20070)
		ld	hl,11059				; 10	; rp += 11059
		zadd	hl,de					; 11	; 20070/32767 = 0.6125 = 1.225/2
		jp	(ReflectRET+3)				; 10	; done

	.No20070:
		cp	a					;  4	; clear carry
		rr	d					;  4	; rp >>= 1
		rr	e					;  4	; ''
		cp	a					;  4	; clear carry
		rr	d					;  4	; rp >>= 1
		rr	e					;  4	; ''
		ld	hl,26112				; 10	; rp = (rp >> 2) + 26112
		zadd	hl,de					; 11	; 26112/32767 = 0.796875
		jp	(ReflectRET+3)				; 10	; done

	.Neg:
		ld	hl,11059-1				; 10	; check if higher than -11059
		zadd	hl,de					; 11	; ''
		jr	nc,.NoM11059				;  7|12	; if (rp <= -11059)
		ex	de,hl					;  4	; rp <<= 1
		zadd	hl,hl					; 11	; 11059/32767 = 0.3375 = 0.675/2
		jp	(ReflectRET+3)				; 10	; done

	.NoM11059:
		ld	hl,20070-1				; 10	; check if higher than -20070
		zadd	hl,de					; 11	; ''
		jr	nc,.NoM20070				;  7|12	; if (rp <= -20070)
		ld	hl,-11059				; 10	; rp += 11059
		zadd	hl,de					; 11	; 20070/32767 = 0.6125 = 1.225/2
		jp	(ReflectRET+3)				; 10	; done

	.NoM20070:
		scf						;  4	; set carry
		rr	d					;  4	; rp >>= 1
		rr	e					;  4	; ''
		scf						;  4	; set carry
		rr	d					;  4	; rp >>= 1
		rr	e					;  4	; ''
		ld	hl,-26112				; 10	; rp = (rp >> 2) + 26112
		zadd	hl,de					; 11	; 26112/32767 = 0.796875
		jp	(ReflectRET+3)				; 10	; done

; ===========================================================================
; ---------------------------------------------------------------------------
; Look up tables for load LARc through arithmetic
; ---------------------------------------------------------------------------
ArithTable	macro	Shift, Mask, AddOff
AD = AddOff
MS = Mask
SH = Shift
	if SH>=0
Byte = 0
		rept	0100h
		db	((Byte>>SH)&MS)+AD
Byte = Byte+1
		endr
	else
SH = -SH
Byte = 0
		rept	0100h
		db	((Byte<<SH)&MS)+AD
Byte = Byte+1
		endr
	endif
		endm
; ---------------------------------------------------------------------------
		align	0100h

LARcTables:

ASL2M00111100:	ArithTable	-2, 00111100b, 0		; DDDD1111 to 00111100
ASR3M00011111:	ArithTable	 3, 00011111b, LARpp3&0FFh	; 33333444 to 00033333
ASL2M00011100:	ArithTable	-2, 00011100b, 0		; 33333444 to 00044400
ASR2M00001111:	ArithTable	 2, 00001111b, LARpp5&0FFh	; 44555566 to 00005555
ASL2M00001100:	ArithTable	-2, 00001100b, 0		; 44555566 to 00006600

; ---------------------------------------------------------------------------
; Look up tables for load xMc through arithmetic
; ---------------------------------------------------------------------------

xMcTables:

ASR4M00000111:	ArithTable	 4, 00000111b, 0		; X1112223 to 00000111 | 8999AAAB to 00000999
ASR1M00000111:	ArithTable	 1, 00000111b, 0		; X1112223 to 00000222 | 8999AAAB to 00000AAA
ASL2M00000100:	ArithTable	-2, 00000100b, 0		; X1112223 to 00000300 | 8999AAAB to 00000B00
ASR6M00000011:	ArithTable	 6, 00000011b, 0		; 11222222 to 00000011 | 44555566 to 00000044 | 66777888 to 00000066 | 33444555 to 00000033 |  | BBCCCDDD to 000000BB
ASR3M00000111:	ArithTable	 3, 00000111b, 0		; 66777888 to 00000777 | 33444555 to 00000444 | BBCCCDDD to 00000CCC
;...zand 00000111b
ASR5M00000111:	ArithTable	 5, 00000111b, 0		; 66677788 to 00000666
ASR2M00000111:	ArithTable	 2, 00000111b, 0		; 66677788 to 00000777
ASL1M00000110:	ArithTable	-1, 00000110b, 0		; 66677788 to 00000880
; Old table version (turns out rotating and anding ended up being the same cycle time).
;ASR7M00000001:	ArithTable	 7, 00000001b, 0		; 8999AAAB to 00000008

; ---------------------------------------------------------------------------
; xMc/xmaxc to ep table value
; ---------------------------------------------------------------------------
;	xMc
;xmaxc	FFE4 FFEC FFF4 FFFC 0004 000C 0014 001C
;	FFC8 FFD8 FFE8 FFF8 0008 0018 0028 0038 
;	FFAC FFC4 FFDC FFF4 000C 0024 003C 0054 
;	FF90 FFB0 FFD0 FFF0 0010 0030 0050 0070 
;	FF74 FF9C FFC4 FFEC 0014 003C 0064 008C 
;	FF58 FF88 FFB8 FFE8 0018 0048 0078 00A8 
;	FF3C FF74 FFAC FFE4 001C 0054 008C 00C4 
;	FF20 FF60 FFA0 FFE0 0020 0060 00A0 00E0 
;	FF04 FF4C FF94 FFDC 0024 006C 00B4 00FC 
;	FEE8 FF38 FF88 FFD8 0028 0078 00C8 0118 
;	FECC FF24 FF7C FFD4 002C 0084 00DC 0134 
;	FEB0 FF10 FF70 FFD0 0030 0090 00F0 0150 
;	FE94 FEFC FF64 FFCC 0034 009C 0104 016C 
;	FE78 FEE8 FF58 FFC8 0038 00A8 0118 0188 
;	FE5C FED4 FF4C FFC4 003C 00B4 012C 01A4 
;	FE40 FEC0 FF40 FFC0 0040 00C0 0140 01C0 
;	FE08 FE98 FF28 FFB8 0048 00D8 0168 01F8 
;	FDD0 FE70 FF10 FFB0 0050 00F0 0190 0230 
;	FD98 FE48 FEF8 FFA8 0058 0108 01B8 0268 
;	FD60 FE20 FEE0 FFA0 0060 0120 01E0 02A0 
;	FD28 FDF8 FEC8 FF98 0068 0138 0208 02D8 
;	FCF0 FDD0 FEB0 FF90 0070 0150 0230 0310 
;	FCB8 FDA8 FE98 FF88 0078 0168 0258 0348 
;	FC80 FD80 FE80 FF80 0080 0180 0280 0380 
;	FC10 FD30 FE50 FF70 0090 01B0 02D0 03F0 
;	FBA0 FCE0 FE20 FF60 00A0 01E0 0320 0460 
;	FB30 FC90 FDF0 FF50 00B0 0210 0370 04D0 
;	FAC0 FC40 FDC0 FF40 00C0 0240 03C0 0540 
;	FA50 FBF0 FD90 FF30 00D0 0270 0410 05B0 
;	F9E0 FBA0 FD60 FF20 00E0 02A0 0460 0620 
;	F970 FB50 FD30 FF10 00F0 02D0 04B0 0690 
;	F900 FB00 FD00 FF00 0100 0300 0500 0700 
;	F820 FA60 FCA0 FEE0 0120 0360 05A0 07E0 
;	F740 F9C0 FC40 FEC0 0140 03C0 0640 08C0 
;	F660 F920 FBE0 FEA0 0160 0420 06E0 09A0 
;	F580 F880 FB80 FE80 0180 0480 0780 0A80 
;	F4A0 F7E0 FB20 FE60 01A0 04E0 0820 0B60 
;	F3C0 F740 FAC0 FE40 01C0 0540 08C0 0C40 
;	F2E0 F6A0 FA60 FE20 01E0 05A0 0960 0D20 
;	F200 F600 FA00 FE00 0200 0600 0A00 0E00 
;	F040 F4C0 F940 FDC0 0240 06C0 0B40 0FC0 
;	EE80 F380 F880 FD80 0280 0780 0C80 1180 
;	ECC0 F240 F7C0 FD40 02C0 0840 0DC0 1340 
;	EB00 F100 F700 FD00 0300 0900 0F00 1500 
;	E940 EFC0 F640 FCC0 0340 09C0 1040 16C0 
;	E780 EE80 F580 FC80 0380 0A80 1180 1880 
;	E5C0 ED40 F4C0 FC40 03C0 0B40 12C0 1A40 
;	E400 EC00 F400 FC00 0400 0C00 1400 1C00 
;	E081 E981 F280 FB80 0480 0D80 1680 1F80 
;	DD01 E701 F100 FB00 0500 0F00 1900 2300 
;	D981 E481 EF80 FA80 0580 1080 1B80 2680 
;	D601 E201 EE00 FA00 0600 1200 1E00 2A00 
;	D281 DF81 EC80 F980 0680 1380 2080 2D80 
;	CF01 DD01 EB00 F900 0700 1500 2300 3100 
;	CB81 DA81 E980 F880 0780 1680 2580 3480 
;	C801 D801 E800 F800 0800 1800 2800 3800 
;	C101 D301 E500 F700 0900 1B00 2CFF 3EFF 
;	BA01 CE01 E200 F600 0A00 1E00 31FF 45FF 
;	B301 C901 DF00 F500 0B00 2100 36FF 4CFF 
;	AC01 C401 DC00 F400 0C00 2400 3BFF 53FF 
;	A501 BF01 D900 F300 0D00 2700 40FF 5AFF 
;	9E01 BA01 D600 F200 0E00 2A00 45FF 61FF 
;	9701 B501 D300 F100 0F00 2D00 4AFF 68FF 
;	9001 B001 D000 F000 1000 3000 4FFF 6FFF 
; ---------------------------------------------------------------------------

	; low byte first
	; xMc =		000  001  010  011  100  101  110  111

epval:		db	0E4h,0ECh,0F4h,0FCh,004h,00Ch,014h,01Ch		; xmaxc = 000000
		db	0C8h,0D8h,0E8h,0F8h,008h,018h,028h,038h		; xmaxc = 000001
		db	0ACh,0C4h,0DCh,0F4h,00Ch,024h,03Ch,054h		; xmaxc = 000010
		db	090h,0B0h,0D0h,0F0h,010h,030h,050h,070h		; xmaxc = 000011
		db	074h,09Ch,0C4h,0ECh,014h,03Ch,064h,08Ch		; xmaxc = 000100
		db	058h,088h,0B8h,0E8h,018h,048h,078h,0A8h		; xmaxc = 000101
		db	03Ch,074h,0ACh,0E4h,01Ch,054h,08Ch,0C4h		; xmaxc = 000110
		db	020h,060h,0A0h,0E0h,020h,060h,0A0h,0E0h		; xmaxc = 000111
		db	004h,04Ch,094h,0DCh,024h,06Ch,0B4h,0FCh		; xmaxc = 001000
		db	0E8h,038h,088h,0D8h,028h,078h,0C8h,018h		; xmaxc = 001001
		db	0CCh,024h,07Ch,0D4h,02Ch,084h,0DCh,034h		; xmaxc = 001010
		db	0B0h,010h,070h,0D0h,030h,090h,0F0h,050h		; xmaxc = 001011
		db	094h,0FCh,064h,0CCh,034h,09Ch,004h,06Ch		; xmaxc = 001100
		db	078h,0E8h,058h,0C8h,038h,0A8h,018h,088h		; xmaxc = 001101
		db	05Ch,0D4h,04Ch,0C4h,03Ch,0B4h,02Ch,0A4h		; xmaxc = 001110
		db	040h,0C0h,040h,0C0h,040h,0C0h,040h,0C0h		; xmaxc = 001111
		db	008h,098h,028h,0B8h,048h,0D8h,068h,0F8h		; xmaxc = 010000
		db	0D0h,070h,010h,0B0h,050h,0F0h,090h,030h		; xmaxc = 010001
		db	098h,048h,0F8h,0A8h,058h,008h,0B8h,068h		; xmaxc = 010010
		db	060h,020h,0E0h,0A0h,060h,020h,0E0h,0A0h		; xmaxc = 010011
		db	028h,0F8h,0C8h,098h,068h,038h,008h,0D8h		; xmaxc = 010100
		db	0F0h,0D0h,0B0h,090h,070h,050h,030h,010h		; xmaxc = 010101
		db	0B8h,0A8h,098h,088h,078h,068h,058h,048h		; xmaxc = 010110
		db	080h,080h,080h,080h,080h,080h,080h,080h		; xmaxc = 010111
		db	010h,030h,050h,070h,090h,0B0h,0D0h,0F0h		; xmaxc = 011000
		db	0A0h,0E0h,020h,060h,0A0h,0E0h,020h,060h		; xmaxc = 011001
		db	030h,090h,0F0h,050h,0B0h,010h,070h,0D0h		; xmaxc = 011010
		db	0C0h,040h,0C0h,040h,0C0h,040h,0C0h,040h		; xmaxc = 011011
		db	050h,0F0h,090h,030h,0D0h,070h,010h,0B0h		; xmaxc = 011100
		db	0E0h,0A0h,060h,020h,0E0h,0A0h,060h,020h		; xmaxc = 011101
		db	070h,050h,030h,010h,0F0h,0D0h,0B0h,090h		; xmaxc = 011110
		db	000h,000h,000h,000h,000h,000h,000h,000h		; xmaxc = 011111

	; high byte next
	; xMc =		000  001  010  011  100  101  110  111

		db	0FFh,0FFh,0FFh,0FFh,000h,000h,000h,000h		; xmaxc = 000000
		db	0FFh,0FFh,0FFh,0FFh,000h,000h,000h,000h		; xmaxc = 000001
		db	0FFh,0FFh,0FFh,0FFh,000h,000h,000h,000h		; xmaxc = 000010
		db	0FFh,0FFh,0FFh,0FFh,000h,000h,000h,000h		; xmaxc = 000011
		db	0FFh,0FFh,0FFh,0FFh,000h,000h,000h,000h		; xmaxc = 000100
		db	0FFh,0FFh,0FFh,0FFh,000h,000h,000h,000h		; xmaxc = 000101
		db	0FFh,0FFh,0FFh,0FFh,000h,000h,000h,000h		; xmaxc = 000110
		db	0FFh,0FFh,0FFh,0FFh,000h,000h,000h,000h		; xmaxc = 000111
		db	0FFh,0FFh,0FFh,0FFh,000h,000h,000h,000h		; xmaxc = 001000
		db	0FEh,0FFh,0FFh,0FFh,000h,000h,000h,001h		; xmaxc = 001001
		db	0FEh,0FFh,0FFh,0FFh,000h,000h,000h,001h		; xmaxc = 001010
		db	0FEh,0FFh,0FFh,0FFh,000h,000h,000h,001h		; xmaxc = 001011
		db	0FEh,0FEh,0FFh,0FFh,000h,000h,001h,001h		; xmaxc = 001100
		db	0FEh,0FEh,0FFh,0FFh,000h,000h,001h,001h		; xmaxc = 001101
		db	0FEh,0FEh,0FFh,0FFh,000h,000h,001h,001h		; xmaxc = 001110
		db	0FEh,0FEh,0FFh,0FFh,000h,000h,001h,001h		; xmaxc = 001111
		db	0FEh,0FEh,0FFh,0FFh,000h,000h,001h,001h		; xmaxc = 010000
		db	0FDh,0FEh,0FFh,0FFh,000h,000h,001h,002h		; xmaxc = 010001
		db	0FDh,0FEh,0FEh,0FFh,000h,001h,001h,002h		; xmaxc = 010010
		db	0FDh,0FEh,0FEh,0FFh,000h,001h,001h,002h		; xmaxc = 010011
		db	0FDh,0FDh,0FEh,0FFh,000h,001h,002h,002h		; xmaxc = 010100
		db	0FCh,0FDh,0FEh,0FFh,000h,001h,002h,003h		; xmaxc = 010101
		db	0FCh,0FDh,0FEh,0FFh,000h,001h,002h,003h		; xmaxc = 010110
		db	0FCh,0FDh,0FEh,0FFh,000h,001h,002h,003h		; xmaxc = 010111
		db	0FCh,0FDh,0FEh,0FFh,000h,001h,002h,003h		; xmaxc = 011000
		db	0FBh,0FCh,0FEh,0FFh,000h,001h,003h,004h		; xmaxc = 011001
		db	0FBh,0FCh,0FDh,0FFh,000h,002h,003h,004h		; xmaxc = 011010
		db	0FAh,0FCh,0FDh,0FFh,000h,002h,003h,005h		; xmaxc = 011011
		db	0FAh,0FBh,0FDh,0FFh,000h,002h,004h,005h		; xmaxc = 011100
		db	0F9h,0FBh,0FDh,0FFh,000h,002h,004h,006h		; xmaxc = 011101
		db	0F9h,0FBh,0FDh,0FFh,000h,002h,004h,006h		; xmaxc = 011110
		db	0F9h,0FBh,0FDh,0FFh,001h,003h,005h,007h		; xmaxc = 011111

	; low byte first
	; xMc =		000  001  010  011  100  101  110  111

		db	020h,060h,0A0h,0E0h,020h,060h,0A0h,0E0h		; xmaxc = 100000
		db	040h,0C0h,040h,0C0h,040h,0C0h,040h,0C0h		; xmaxc = 100001
		db	060h,020h,0E0h,0A0h,060h,020h,0E0h,0A0h		; xmaxc = 100010
		db	080h,080h,080h,080h,080h,080h,080h,080h		; xmaxc = 100011
		db	0A0h,0E0h,020h,060h,0A0h,0E0h,020h,060h		; xmaxc = 100100
		db	0C0h,040h,0C0h,040h,0C0h,040h,0C0h,040h		; xmaxc = 100101
		db	0E0h,0A0h,060h,020h,0E0h,0A0h,060h,020h		; xmaxc = 100110
		db	000h,000h,000h,000h,000h,000h,000h,000h		; xmaxc = 100111
		db	040h,0C0h,040h,0C0h,040h,0C0h,040h,0C0h		; xmaxc = 101000
		db	080h,080h,080h,080h,080h,080h,080h,080h		; xmaxc = 101001
		db	0C0h,040h,0C0h,040h,0C0h,040h,0C0h,040h		; xmaxc = 101010
		db	000h,000h,000h,000h,000h,000h,000h,000h		; xmaxc = 101011
		db	040h,0C0h,040h,0C0h,040h,0C0h,040h,0C0h		; xmaxc = 101100
		db	080h,080h,080h,080h,080h,080h,080h,080h		; xmaxc = 101101
		db	0C0h,040h,0C0h,040h,0C0h,040h,0C0h,040h		; xmaxc = 101110
		db	000h,000h,000h,000h,000h,000h,000h,000h		; xmaxc = 101111
		db	081h,081h,080h,080h,080h,080h,080h,080h		; xmaxc = 110000
		db	001h,001h,000h,000h,000h,000h,000h,000h		; xmaxc = 110001
		db	081h,081h,080h,080h,080h,080h,080h,080h		; xmaxc = 110010
		db	001h,001h,000h,000h,000h,000h,000h,000h		; xmaxc = 110011
		db	081h,081h,080h,080h,080h,080h,080h,080h		; xmaxc = 110100
		db	001h,001h,000h,000h,000h,000h,000h,000h		; xmaxc = 110101
		db	081h,081h,080h,080h,080h,080h,080h,080h		; xmaxc = 110110
		db	001h,001h,000h,000h,000h,000h,000h,000h		; xmaxc = 110111
		db	001h,001h,000h,000h,000h,000h,0FFh,0FFh		; xmaxc = 111000
		db	001h,001h,000h,000h,000h,000h,0FFh,0FFh		; xmaxc = 111001
		db	001h,001h,000h,000h,000h,000h,0FFh,0FFh		; xmaxc = 111010
		db	001h,001h,000h,000h,000h,000h,0FFh,0FFh		; xmaxc = 111011
		db	001h,001h,000h,000h,000h,000h,0FFh,0FFh		; xmaxc = 111100
		db	001h,001h,000h,000h,000h,000h,0FFh,0FFh		; xmaxc = 111101
		db	001h,001h,000h,000h,000h,000h,0FFh,0FFh		; xmaxc = 111110
		db	001h,001h,000h,000h,000h,000h,0FFh,0FFh		; xmaxc = 111111

	; high byte next
	; xMc =		000  001  010  011  100  101  110  111

		db	0F8h,0FAh,0FCh,0FEh,001h,003h,005h,007h		; xmaxc = 100000
		db	0F7h,0F9h,0FCh,0FEh,001h,003h,006h,008h		; xmaxc = 100001
		db	0F6h,0F9h,0FBh,0FEh,001h,004h,006h,009h		; xmaxc = 100010
		db	0F5h,0F8h,0FBh,0FEh,001h,004h,007h,00Ah		; xmaxc = 100011
		db	0F4h,0F7h,0FBh,0FEh,001h,004h,008h,00Bh		; xmaxc = 100100
		db	0F3h,0F7h,0FAh,0FEh,001h,005h,008h,00Ch		; xmaxc = 100101
		db	0F2h,0F6h,0FAh,0FEh,001h,005h,009h,00Dh		; xmaxc = 100110
		db	0F2h,0F6h,0FAh,0FEh,002h,006h,00Ah,00Eh		; xmaxc = 100111
		db	0F0h,0F4h,0F9h,0FDh,002h,006h,00Bh,00Fh		; xmaxc = 101000
		db	0EEh,0F3h,0F8h,0FDh,002h,007h,00Ch,011h		; xmaxc = 101001
		db	0ECh,0F2h,0F7h,0FDh,002h,008h,00Dh,013h		; xmaxc = 101010
		db	0EBh,0F1h,0F7h,0FDh,003h,009h,00Fh,015h		; xmaxc = 101011
		db	0E9h,0EFh,0F6h,0FCh,003h,009h,010h,016h		; xmaxc = 101100
		db	0E7h,0EEh,0F5h,0FCh,003h,00Ah,011h,018h		; xmaxc = 101101
		db	0E5h,0EDh,0F4h,0FCh,003h,00Bh,012h,01Ah		; xmaxc = 101110
		db	0E4h,0ECh,0F4h,0FCh,004h,00Ch,014h,01Ch		; xmaxc = 101111
		db	0E0h,0E9h,0F2h,0FBh,004h,00Dh,016h,01Fh		; xmaxc = 110000
		db	0DDh,0E7h,0F1h,0FBh,005h,00Fh,019h,023h		; xmaxc = 110001
		db	0D9h,0E4h,0EFh,0FAh,005h,010h,01Bh,026h		; xmaxc = 110010
		db	0D6h,0E2h,0EEh,0FAh,006h,012h,01Eh,02Ah		; xmaxc = 110011
		db	0D2h,0DFh,0ECh,0F9h,006h,013h,020h,02Dh		; xmaxc = 110100
		db	0CFh,0DDh,0EBh,0F9h,007h,015h,023h,031h		; xmaxc = 110101
		db	0CBh,0DAh,0E9h,0F8h,007h,016h,025h,034h		; xmaxc = 110110
		db	0C8h,0D8h,0E8h,0F8h,008h,018h,028h,038h		; xmaxc = 110111
		db	0C1h,0D3h,0E5h,0F7h,009h,01Bh,02Ch,03Eh		; xmaxc = 111000
		db	0BAh,0CEh,0E2h,0F6h,00Ah,01Eh,031h,045h		; xmaxc = 111001
		db	0B3h,0C9h,0DFh,0F5h,00Bh,021h,036h,04Ch		; xmaxc = 111010
		db	0ACh,0C4h,0DCh,0F4h,00Ch,024h,03Bh,053h		; xmaxc = 111011
		db	0A5h,0BFh,0D9h,0F3h,00Dh,027h,040h,05Ah		; xmaxc = 111100
		db	09Eh,0BAh,0D6h,0F2h,00Eh,02Ah,045h,061h		; xmaxc = 111101
		db	097h,0B5h,0D3h,0F1h,00Fh,02Dh,04Ah,068h		; xmaxc = 111110
		db	090h,0B0h,0D0h,0F0h,010h,030h,04Fh,06Fh		; xmaxc = 111111

; ===========================================================================
; ---------------------------------------------------------------------------
; c x 6E14 (low byte) [high bytes is always 00]
; negate low bytes (and high byte 00) if b is negative
; ---------------------------------------------------------------------------
; Doesn't work 100%, sometimes is off by 1 on negative numbers (due to rounding)
; ---------------------------------------------------------------------------

MULTR6E14:

		; --- c x 6E14 (when 0000 to 00FF) [negate when upper is negative] ---

	;	db	000h, 001h, 002h, 003h, 003h, 004h, 005h, 006h, 007h, 008h, 009h, 009h, 00Ah, 00Bh, 00Ch, 00Dh
	;	db	00Eh, 00Fh, 00Fh, 010h, 011h, 012h, 013h, 014h, 015h, 015h, 016h, 017h, 018h, 019h, 01Ah, 01Bh
	;	db	01Ch, 01Ch, 01Dh, 01Eh, 01Fh, 020h, 021h, 022h, 022h, 023h, 024h, 025h, 026h, 027h, 028h, 028h
	;	db	029h, 02Ah, 02Bh, 02Ch, 02Dh, 02Eh, 02Eh, 02Fh, 030h, 031h, 032h, 033h, 034h, 034h, 035h, 036h
	;	db	037h, 038h, 039h, 03Ah, 03Ah, 03Bh, 03Ch, 03Dh, 03Eh, 03Fh, 040h, 040h, 041h, 042h, 043h, 044h
	;	db	045h, 046h, 047h, 047h, 048h, 049h, 04Ah, 04Bh, 04Ch, 04Dh, 04Dh, 04Eh, 04Fh, 050h, 051h, 052h
	;	db	053h, 053h, 054h, 055h, 056h, 057h, 058h, 059h, 059h, 05Ah, 05Bh, 05Ch, 05Dh, 05Eh, 05Fh, 05Fh
	;	db	060h, 061h, 062h, 063h, 064h, 065h, 065h, 066h, 067h, 068h, 069h, 06Ah, 06Bh, 06Bh, 06Ch, 06Dh
	;	db	06Eh, 06Fh, 070h, 071h, 072h, 072h, 073h, 074h, 075h, 076h, 077h, 078h, 078h, 079h, 07Ah, 07Bh
	;	db	07Ch, 07Dh, 07Eh, 07Eh, 07Fh, 080h, 081h, 082h, 083h, 084h, 084h, 085h, 086h, 087h, 088h, 089h
	;	db	08Ah, 08Ah, 08Bh, 08Ch, 08Dh, 08Eh, 08Fh, 090h, 090h, 091h, 092h, 093h, 094h, 095h, 096h, 096h
	;	db	097h, 098h, 099h, 09Ah, 09Bh, 09Ch, 09Dh, 09Dh, 09Eh, 09Fh, 0A0h, 0A1h, 0A2h, 0A3h, 0A3h, 0A4h
	;	db	0A5h, 0A6h, 0A7h, 0A8h, 0A9h, 0A9h, 0AAh, 0ABh, 0ACh, 0ADh, 0AEh, 0AFh, 0AFh, 0B0h, 0B1h, 0B2h
	;	db	0B3h, 0B4h, 0B5h, 0B5h, 0B6h, 0B7h, 0B8h, 0B9h, 0BAh, 0BBh, 0BBh, 0BCh, 0BDh, 0BEh, 0BFh, 0C0h
	;	db	0C1h, 0C1h, 0C2h, 0C3h, 0C4h, 0C5h, 0C6h, 0C7h, 0C8h, 0C8h, 0C9h, 0CAh, 0CBh, 0CCh, 0CDh, 0CEh
	;	db	0CEh, 0CFh, 0D0h, 0D1h, 0D2h, 0D3h, 0D4h, 0D4h, 0D5h, 0D6h, 0D7h, 0D8h, 0D9h, 0DAh, 0DAh, 0DBh

		; --- b x 6E14 (low byte) ---

	;	db	000h, 0DCh, 0B8h, 094h, 071h, 04Dh, 029h, 005h, 0E1h, 0BDh, 09Ah, 076h, 052h, 02Eh, 00Ah, 0E6h
	;	db	0C3h, 09Fh, 07Bh, 057h, 033h, 00Fh, 0EBh, 0C8h, 0A4h, 080h, 05Ch, 038h, 014h, 0F1h, 0CDh, 0A9h
	;	db	085h, 061h, 03Dh, 019h, 0F6h, 0D2h, 0AEh, 08Ah, 066h, 042h, 01Fh, 0FBh, 0D7h, 0B3h, 08Fh, 06Bh
	;	db	048h, 024h, 000h, 0DCh, 0B8h, 094h, 070h, 04Dh, 029h, 005h, 0E1h, 0BDh, 099h, 076h, 052h, 02Eh
	;	db	00Ah, 0E6h, 0C2h, 09Eh, 07Bh, 057h, 033h, 00Fh, 0EBh, 0C7h, 0A4h, 080h, 05Ch, 038h, 014h, 0F0h
	;	db	0CDh, 0A9h, 085h, 061h, 03Dh, 019h, 0F5h, 0D2h, 0AEh, 08Ah, 066h, 042h, 01Eh, 0FBh, 0D7h, 0B3h
	;	db	08Fh, 06Bh, 047h, 023h, 000h, 0DCh, 0B8h, 094h, 070h, 04Ch, 029h, 005h, 0E1h, 0BDh, 099h, 075h
	;	db	052h, 02Eh, 00Ah, 0E6h, 0C2h, 09Eh, 07Ah, 057h, 033h, 00Fh, 0EBh, 0C7h, 0A3h, 080h, 05Ch, 038h
	;	db	0ECh, 0C8h, 0A4h, 080h, 05Dh, 039h, 015h, 0F1h, 0CDh, 0A9h, 086h, 062h, 03Eh, 01Ah, 0F6h, 0D2h
	;	db	0AFh, 08Bh, 067h, 043h, 01Fh, 0FBh, 0D7h, 0B4h, 090h, 06Ch, 048h, 024h, 000h, 0DDh, 0B9h, 095h
	;	db	071h, 04Dh, 029h, 005h, 0E2h, 0BEh, 09Ah, 076h, 052h, 02Eh, 00Bh, 0E7h, 0C3h, 09Fh, 07Bh, 057h
	;	db	034h, 010h, 0ECh, 0C8h, 0A4h, 080h, 05Ch, 039h, 015h, 0F1h, 0CDh, 0A9h, 085h, 062h, 03Eh, 01Ah
	;	db	0F6h, 0D2h, 0AEh, 08Ah, 067h, 043h, 01Fh, 0FBh, 0D7h, 0B3h, 090h, 06Ch, 048h, 024h, 000h, 0DCh
	;	db	0B9h, 095h, 071h, 04Dh, 029h, 005h, 0E1h, 0BEh, 09Ah, 076h, 052h, 02Eh, 00Ah, 0E7h, 0C3h, 09Fh
	;	db	07Bh, 057h, 033h, 00Fh, 0ECh, 0C8h, 0A4h, 080h, 05Ch, 038h, 015h, 0F1h, 0CDh, 0A9h, 085h, 061h
	;	db	03Eh, 01Ah, 0F6h, 0D2h, 0AEh, 08Ah, 066h, 043h, 01Fh, 0FBh, 0D7h, 0B3h, 08Fh, 06Ch, 048h, 024h

		; --- b x 6E14 (high byte) ---

	;	db	000h, 000h, 001h, 002h, 003h, 004h, 005h, 006h, 006h, 007h, 008h, 009h, 00Ah, 00Bh, 00Ch, 00Ch
	;	db	00Dh, 00Eh, 00Fh, 010h, 011h, 012h, 012h, 013h, 014h, 015h, 016h, 017h, 018h, 018h, 019h, 01Ah
	;	db	01Bh, 01Ch, 01Dh, 01Eh, 01Eh, 01Fh, 020h, 021h, 022h, 023h, 024h, 024h, 025h, 026h, 027h, 028h
	;	db	029h, 02Ah, 02Bh, 02Bh, 02Ch, 02Dh, 02Eh, 02Fh, 030h, 031h, 031h, 032h, 033h, 034h, 035h, 036h
	;	db	037h, 037h, 038h, 039h, 03Ah, 03Bh, 03Ch, 03Dh, 03Dh, 03Eh, 03Fh, 040h, 041h, 042h, 043h, 043h
	;	db	044h, 045h, 046h, 047h, 048h, 049h, 049h, 04Ah, 04Bh, 04Ch, 04Dh, 04Eh, 04Fh, 04Fh, 050h, 051h
	;	db	052h, 053h, 054h, 055h, 056h, 056h, 057h, 058h, 059h, 05Ah, 05Bh, 05Ch, 05Ch, 05Dh, 05Eh, 05Fh
	;	db	060h, 061h, 062h, 062h, 063h, 064h, 065h, 066h, 067h, 068h, 068h, 069h, 06Ah, 06Bh, 06Ch, 06Dh
	;	db	091h, 092h, 093h, 094h, 095h, 096h, 097h, 097h, 098h, 099h, 09Ah, 09Bh, 09Ch, 09Dh, 09Dh, 09Eh
	;	db	09Fh, 0A0h, 0A1h, 0A2h, 0A3h, 0A3h, 0A4h, 0A5h, 0A6h, 0A7h, 0A8h, 0A9h, 0AAh, 0AAh, 0ABh, 0ACh
	;	db	0ADh, 0AEh, 0AFh, 0B0h, 0B0h, 0B1h, 0B2h, 0B3h, 0B4h, 0B5h, 0B6h, 0B6h, 0B7h, 0B8h, 0B9h, 0BAh
	;	db	0BBh, 0BCh, 0BCh, 0BDh, 0BEh, 0BFh, 0C0h, 0C1h, 0C2h, 0C2h, 0C3h, 0C4h, 0C5h, 0C6h, 0C7h, 0C8h
	;	db	0C8h, 0C9h, 0CAh, 0CBh, 0CCh, 0CDh, 0CEh, 0CEh, 0CFh, 0D0h, 0D1h, 0D2h, 0D3h, 0D4h, 0D5h, 0D5h
	;	db	0D6h, 0D7h, 0D8h, 0D9h, 0DAh, 0DBh, 0DBh, 0DCh, 0DDh, 0DEh, 0DFh, 0E0h, 0E1h, 0E1h, 0E2h, 0E3h
	;	db	0E4h, 0E5h, 0E6h, 0E7h, 0E7h, 0E8h, 0E9h, 0EAh, 0EBh, 0ECh, 0EDh, 0EDh, 0EEh, 0EFh, 0F0h, 0F1h
	;	db	0F2h, 0F3h, 0F3h, 0F4h, 0F5h, 0F6h, 0F7h, 0F8h, 0F9h, 0F9h, 0FAh, 0FBh, 0FCh, 0FDh, 0FEh, 0FFh

; ===========================================================================















